use wolfssl::wolfcrypt::sha::*;

#[test]
#[cfg(sha)]
fn test_sha() {
    let mut sha = SHA::new().expect("Error with new()");
    fn test1(sha: &mut SHA, input: &[u8], expected_hash: &[u8]) {
        sha.init().expect("Error with init()");
        sha.update(input).expect("Error with update()");
        let mut hash = [0u8; SHA::DIGEST_SIZE];
        sha.finalize(&mut hash).expect("Error with finalize()");
        assert_eq!(hash, *expected_hash);
    }

    test1(&mut sha, b"", b"\xda\x39\xa3\xee\x5e\x6b\x4b\x0d\x32\x55\xbf\xef\x95\x60\x18\x90\xaf\xd8\x07\x09");

    test1(&mut sha, b"abc", b"\xA9\x99\x3E\x36\x47\x06\x81\x6A\xBA\x3E\x25\x71\x78\x50\xC2\x6C\x9C\xD0\xD8\x9D");

    test1(&mut sha, b"abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
        b"\x84\x98\x3E\x44\x1C\x3B\xD2\x6E\xBA\xAE\x4A\xA1\xF9\x51\x29\xE5\xE5\x46\x70\xF1");

    test1(&mut sha, b"aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa",
        b"\x00\x98\xBA\x82\x4B\x5C\x16\x42\x7B\xD7\xA1\x12\x2A\x5A\x44\x2A\x25\xEC\x64\x4D");

    test1(&mut sha, b"aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa",
        b"\xAD\x5B\x3F\xDB\xCB\x52\x67\x78\xC2\x83\x9D\x2F\x15\x1E\xA7\x53\x99\x5E\x26\xA0");
}

#[test]
#[cfg(sha256)]
fn test_sha224() {
    let mut sha = SHA224::new().expect("Error with new()");
    fn test1(sha: &mut SHA224, input: &[u8], expected_hash: &[u8]) {
        sha.init().expect("Error with init()");
        sha.update(input).expect("Error with update()");
        let mut hash = [0u8; SHA224::DIGEST_SIZE];
        sha.finalize(&mut hash).expect("Error with finalize()");
        assert_eq!(hash, *expected_hash);
    }

    test1(&mut sha, b"", b"\xd1\x4a\x02\x8c\x2a\x3a\x2b\xc9\x47\x61\x02\xbb\x28\x82\x34\xc4\x15\xa2\xb0\x1f\x82\x8e\xa6\x2a\xc5\xb3\xe4\x2f");

    test1(&mut sha, b"abc", b"\x23\x09\x7d\x22\x34\x05\xd8\x22\x86\x42\xa4\x77\xbd\xa2\x55\xb3\x2a\xad\xbc\xe4\xbd\xa0\xb3\xf7\xe3\x6c\x9d\xa7");

    test1(&mut sha, b"abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
        b"\x75\x38\x8b\x16\x51\x27\x76\xcc\x5d\xba\x5d\xa1\xfd\x89\x01\x50\xb0\xc6\x45\x5c\xb4\xf5\x8b\x19\x52\x52\x25\x25");
}

#[test]
#[cfg(sha256)]
fn test_sha256() {
    let mut sha = SHA256::new().expect("Error with new()");
    fn test1(sha: &mut SHA256, input: &[u8], expected_hash: &[u8]) {
        sha.init().expect("Error with init()");
        sha.update(input).expect("Error with update()");
        let mut hash = [0u8; SHA256::DIGEST_SIZE];
        sha.finalize(&mut hash).expect("Error with finalize()");
        assert_eq!(hash, *expected_hash);
    }

    test1(&mut sha, b"",
        b"\xe3\xb0\xc4\x42\x98\xfc\x1c\x14\x9a\xfb\xf4\xc8\x99\x6f\xb9\x24\x27\xae\x41\xe4\x64\x9b\x93\x4c\xa4\x95\x99\x1b\x78\x52\xb8\x55");

    test1(&mut sha, b"abc",
        b"\xBA\x78\x16\xBF\x8F\x01\xCF\xEA\x41\x41\x40\xDE\x5D\xAE\x22\x23\xB0\x03\x61\xA3\x96\x17\x7A\x9C\xB4\x10\xFF\x61\xF2\x00\x15\xAD");

    test1(&mut sha, b"abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
        b"\x24\x8D\x6A\x61\xD2\x06\x38\xB8\xE5\xC0\x26\x93\x0C\x3E\x60\x39\xA3\x3C\xE4\x59\x64\xFF\x21\x67\xF6\xEC\xED\xD4\x19\xDB\x06\xC1");

    test1(&mut sha,
        b"aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", /* this is BLOCKSIZE length */
        b"\xFF\xE0\x54\xFE\x7A\xE0\xCB\x6D\xC6\x5C\x3A\xF9\xB6\x1D\x52\x09\xF4\x39\x85\x1D\xB4\x3D\x0B\xA5\x99\x73\x37\xDF\x15\x46\x68\xEB");
}

#[test]
#[cfg(sha512)]
fn test_sha384() {
    let mut sha = SHA384::new().expect("Error with new()");
    fn test1(sha: &mut SHA384, input: &[u8], expected_hash: &[u8]) {
        sha.init().expect("Error with init()");
        sha.update(input).expect("Error with update()");
        let mut hash = [0u8; SHA384::DIGEST_SIZE];
        sha.finalize(&mut hash).expect("Error with finalize()");
        assert_eq!(hash, *expected_hash);
    }

    test1(&mut sha, b"",
        b"\x38\xb0\x60\xa7\x51\xac\x96\x38\x4c\xd9\x32\x7e\xb1\xb1\xe3\x6a\x21\xfd\xb7\x11\x14\xbe\x07\x43\x4c\x0c\xc7\xbf\x63\xf6\xe1\xda\x27\x4e\xde\xbf\xe7\x6f\x65\xfb\xd5\x1a\xd2\xf1\x48\x98\xb9\x5b");

    test1(&mut sha, b"abc",
        b"\xcb\x00\x75\x3f\x45\xa3\x5e\x8b\xb5\xa0\x3d\x69\x9a\xc6\x50\x07\x27\x2c\x32\xab\x0e\xde\xd1\x63\x1a\x8b\x60\x5a\x43\xff\x5b\xed\x80\x86\x07\x2b\xa1\xe7\xcc\x23\x58\xba\xec\xa1\x34\xc8\x25\xa7");

    test1(&mut sha,
        b"abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu",
        b"\x09\x33\x0c\x33\xf7\x11\x47\xe8\x3d\x19\x2f\xc7\x82\xcd\x1b\x47\x53\x11\x1b\x17\x3b\x3b\x05\xd2\x2f\xa0\x80\x86\xe3\xb0\xf7\x12\xfc\xc7\xc7\x1a\x55\x7e\x2d\xb9\x66\xc3\xe9\xfa\x91\x74\x60\x39");
}

#[test]
#[cfg(sha512)]
fn test_sha512() {
    let mut sha = SHA512::new().expect("Error with new()");
    fn test1(sha: &mut SHA512, input: &[u8], expected_hash: &[u8]) {
        sha.init().expect("Error with init()");
        sha.update(input).expect("Error with update()");
        let mut hash = [0u8; SHA512::DIGEST_SIZE];
        sha.finalize(&mut hash).expect("Error with finalize()");
        assert_eq!(hash, *expected_hash);
    }

    test1(&mut sha, b"",
        b"\xcf\x83\xe1\x35\x7e\xef\xb8\xbd\xf1\x54\x28\x50\xd6\x6d\x80\x07\xd6\x20\xe4\x05\x0b\x57\x15\xdc\x83\xf4\xa9\x21\xd3\x6c\xe9\xce\x47\xd0\xd1\x3c\x5d\x85\xf2\xb0\xff\x83\x18\xd2\x87\x7e\xec\x2f\x63\xb9\x31\xbd\x47\x41\x7a\x81\xa5\x38\x32\x7a\xf9\x27\xda\x3e");

    test1(&mut sha, b"abc",
        b"\xdd\xaf\x35\xa1\x93\x61\x7a\xba\xcc\x41\x73\x49\xae\x20\x41\x31\x12\xe6\xfa\x4e\x89\xa9\x7e\xa2\x0a\x9e\xee\xe6\x4b\x55\xd3\x9a\x21\x92\x99\x2a\x27\x4f\xc1\xa8\x36\xba\x3c\x23\xa3\xfe\xeb\xbd\x45\x4d\x44\x23\x64\x3c\xe8\x0e\x2a\x9a\xc9\x4f\xa5\x4c\xa4\x9f");

    test1(&mut sha,
        b"abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu",
        b"\x8e\x95\x9b\x75\xda\xe3\x13\xda\x8c\xf4\xf7\x28\x14\xfc\x14\x3f\x8f\x77\x79\xc6\xeb\x9f\x7f\xa1\x72\x99\xae\xad\xb6\x88\x90\x18\x50\x1d\x28\x9e\x49\x00\xf7\xe4\x33\x1b\x99\xde\xc4\xb5\x43\x3a\xc7\xd3\x29\xee\xb6\xdd\x26\x54\x5e\x96\xe5\x5b\x87\x4b\xe9\x09");
}

#[test]
#[cfg(sha3)]
fn test_sha3_224() {
    let mut sha = SHA3_224::new().expect("Error with new()");
    fn test1(sha: &mut SHA3_224, input: &[u8], expected_hash: &[u8]) {
        sha.init().expect("Error with init()");
        sha.update(input).expect("Error with update()");
        let mut hash = [0u8; SHA3_224::DIGEST_SIZE];
        sha.finalize(&mut hash).expect("Error with finalize()");
        assert_eq!(hash, *expected_hash);
    }

    test1(&mut sha, b"",
        b"\x6b\x4e\x03\x42\x36\x67\xdb\xb7\x3b\x6e\x15\x45\x4f\x0e\xb1\xab\xd4\x59\x7f\x9a\x1b\x07\x8e\x3f\x5b\x5a\x6b\xc7");

    test1(&mut sha, b"abc",
        b"\xe6\x42\x82\x4c\x3f\x8c\xf2\x4a\xd0\x92\x34\xee\x7d\x3c\x76\x6f\xc9\xa3\xa5\x16\x8d\x0c\x94\xad\x73\xb4\x6f\xdf");

    test1(&mut sha,
        b"abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
        b"\x8a\x24\x10\x8b\x15\x4a\xda\x21\xc9\xfd\x55\x74\x49\x44\x79\xba\x5c\x7e\x7a\xb7\x6e\xf2\x64\xea\xd0\xfc\xce\x33");
}

#[test]
#[cfg(sha3)]
fn test_sha3_256() {
    let mut sha = SHA3_256::new().expect("Error with new()");
    fn test1(sha: &mut SHA3_256, input: &[u8], expected_hash: &[u8]) {
        sha.init().expect("Error with init()");
        sha.update(input).expect("Error with update()");
        let mut hash = [0u8; SHA3_256::DIGEST_SIZE];
        sha.finalize(&mut hash).expect("Error with finalize()");
        assert_eq!(hash, *expected_hash);
    }

    test1(&mut sha, b"",
        b"\xa7\xff\xc6\xf8\xbf\x1e\xd7\x66\x51\xc1\x47\x56\xa0\x61\xd6\x62\xf5\x80\xff\x4d\xe4\x3b\x49\xfa\x82\xd8\x0a\x4b\x80\xf8\x43\x4a");

    test1(&mut sha, b"abc",
        b"\x3a\x98\x5d\xa7\x4f\xe2\x25\xb2\x04\x5c\x17\x2d\x6b\xd3\x90\xbd\x85\x5f\x08\x6e\x3e\x9d\x52\x5b\x46\xbf\xe2\x45\x11\x43\x15\x32");

    test1(&mut sha,
        b"abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
        b"\x41\xc0\xdb\xa2\xa9\xd6\x24\x08\x49\x10\x03\x76\xa8\x23\x5e\x2c\x82\xe1\xb9\x99\x8a\x99\x9e\x21\xdb\x32\xdd\x97\x49\x6d\x33\x76");
}

#[test]
#[cfg(sha3)]
fn test_sha3_384() {
    let mut sha = SHA3_384::new().expect("Error with new()");
    fn test1(sha: &mut SHA3_384, input: &[u8], expected_hash: &[u8]) {
        sha.init().expect("Error with init()");
        sha.update(input).expect("Error with update()");
        let mut hash = [0u8; SHA3_384::DIGEST_SIZE];
        sha.finalize(&mut hash).expect("Error with finalize()");
        assert_eq!(hash, *expected_hash);
    }

    test1(&mut sha, b"",
        b"\x0c\x63\xa7\x5b\x84\x5e\x4f\x7d\x01\x10\x7d\x85\x2e\x4c\x24\x85\xc5\x1a\x50\xaa\xaa\x94\xfc\x61\x99\x5e\x71\xbb\xee\x98\x3a\x2a\xc3\x71\x38\x31\x26\x4a\xdb\x47\xfb\x6b\xd1\xe0\x58\xd5\xf0\x04");

    test1(&mut sha,
        b"\x7d\x80\xb1\x60\xc4\xb5\x36\xa3\xbe\xb7\x99\x80\x59\x93\x44\x04\x7c\x5f\x82\xa1\xdf\xc3\xee\xd4",
        b"\x04\x1c\xc5\x86\x1b\xa3\x34\x56\x3c\x61\xd4\xef\x97\x10\xd4\x89\x6c\x31\x1c\x92\xed\xbe\x0d\x7c\xd5\x3e\x80\x3b\xf2\xf4\xeb\x60\x57\x23\x55\x70\x77\x0c\xe8\x7c\x55\x20\xd7\xec\x14\x19\x87\x22");

    test1(&mut sha, b"abc",
        b"\xec\x01\x49\x82\x88\x51\x6f\xc9\x26\x45\x9f\x58\xe2\xc6\xad\x8d\xf9\xb4\x73\xcb\x0f\xc0\x8c\x25\x96\xda\x7c\xf0\xe4\x9b\xe4\xb2\x98\xd8\x8c\xea\x92\x7a\xc7\xf5\x39\xf1\xed\xf2\x28\x37\x6d\x25");

    test1(&mut sha,
        b"abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
        b"\x99\x1c\x66\x57\x55\xeb\x3a\x4b\x6b\xbd\xfb\x75\xc7\x8a\x49\x2e\x8c\x56\xa2\x2c\x5c\x4d\x7e\x42\x9b\xfd\xbc\x32\xb9\xd4\xad\x5a\xa0\x4a\x1f\x07\x6e\x62\xfe\xa1\x9e\xef\x51\xac\xd0\x65\x7c\x22");
}

#[test]
#[cfg(sha3)]
fn test_sha3_512() {
    let mut sha = SHA3_512::new().expect("Error with new()");
    fn test1(sha: &mut SHA3_512, input: &[u8], expected_hash: &[u8]) {
        sha.init().expect("Error with init()");
        sha.update(input).expect("Error with update()");
        let mut hash = [0u8; SHA3_512::DIGEST_SIZE];
        sha.finalize(&mut hash).expect("Error with finalize()");
        assert_eq!(hash, *expected_hash);
    }

    test1(&mut sha, b"",
        b"\xa6\x9f\x73\xcc\xa2\x3a\x9a\xc5\xc8\xb5\x67\xdc\x18\x5a\x75\x6e\x97\xc9\x82\x16\x4f\xe2\x58\x59\xe0\xd1\xdc\xc1\x47\x5c\x80\xa6\x15\xb2\x12\x3a\xf1\xf5\xf9\x4c\x11\xe3\xe9\x40\x2c\x3a\xc5\x58\xf5\x00\x19\x9d\x95\xb6\xd3\xe3\x01\x75\x85\x86\x28\x1d\xcd\x26");

    test1(&mut sha, b"abc",
        b"\xb7\x51\x85\x0b\x1a\x57\x16\x8a\x56\x93\xcd\x92\x4b\x6b\x09\x6e\x08\xf6\x21\x82\x74\x44\xf7\x0d\x88\x4f\x5d\x02\x40\xd2\x71\x2e\x10\xe1\x16\xe9\x19\x2a\xf3\xc9\x1a\x7e\xc5\x76\x47\xe3\x93\x40\x57\x34\x0b\x4c\xf4\x08\xd5\xa5\x65\x92\xf8\x27\x4e\xec\x53\xf0");

    test1(&mut sha,
        b"abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
        b"\x04\xa3\x71\xe8\x4e\xcf\xb5\xb8\xb7\x7c\xb4\x86\x10\xfc\xa8\x18\x2d\xd4\x57\xce\x6f\x32\x6a\x0f\xd3\xd7\xec\x2f\x1e\x91\x63\x6d\xee\x69\x1f\xbe\x0c\x98\x53\x02\xba\x1b\x0d\x8d\xc7\x8c\x08\x63\x46\xb5\x33\xb4\x9c\x03\x0d\x99\xa2\x7d\xaf\x11\x39\xd6\xe7\x5e");
}

#[test]
#[cfg(shake128)]
fn test_shake128() {
    let mut sha = SHAKE128::new().expect("Error with new()");
    fn test1(sha: &mut SHAKE128, input: &[u8], expected_hash: &[u8]) {
        sha.init().expect("Error with init()");
        sha.update(input).expect("Error with update()");
        let mut hash = vec![0u8; expected_hash.len()];
        sha.finalize(&mut hash).expect("Error with finalize()");
        assert_eq!(hash, *expected_hash);
    }

    test1(&mut sha, b"",
        b"\x7f\x9c\x2b\xa4\xe8\x8f\x82\x7d\x61\x60\x45\x50\x76\x05\x85\x3e\xd7\x3b\x80\x93\xf6\xef\xbc\x88\xeb\x1a\x6e\xac\xfa\x66\xef\x26\x3c\xb1\xee\xa9\x88\x00\x4b\x93\x10\x3c\xfb\x0a\xee\xfd\x2a\x68\x6e\x01\xfa\x4a\x58\xe8\xa3\x63\x9c\xa8\xa1\xe3\xf9\xae\x57\xe2\x35\xb8\xcc\x87\x3c\x23\xdc\x62\xb8\xd2\x60\x16\x9a\xfa\x2f\x75\xab\x91\x6a\x58\xd9\x74\x91\x88\x35\xd2\x5e\x6a\x43\x50\x85\xb2\xba\xdf\xd6\xdf\xaa\xc3\x59\xa5\xef\xbb\x7b\xcc\x4b\x59\xd5\x38\xdf\x9a");

    test1(&mut sha, b"abc",
        b"\x58\x81\x09\x2d\xd8\x18\xbf\x5c\xf8\xa3\xdd\xb7\x93\xfb\xcb\xa7\x40\x97\xd5\xc5\x26\xa6\xd3\x5f\x97\xb8\x33\x51\x94\x0f\x2c\xc8\x44\xc5\x0a\xf3\x2a\xcd\x3f\x2c\xdd\x06\x65\x68\x70\x6f\x50\x9b\xc1\xbd\xde\x58\x29\x5d\xae\x3f\x89\x1a\x9a\x0f\xca\x57\x83\x78\x9a\x41\xf8\x61\x12\x14\xce\x61\x23\x94\xdf\x28\x6a\x62\xd1\xa2\x25\x2a\xa9\x4d\xb9\xc5\x38\x95\x6c\x71\x7d\xc2\xbe\xd4\xf2\x32\xa0\x29\x4c\x85\x7c\x73\x0a\xa1\x60\x67\xac\x10\x62\xf1\x20\x1f\xb0\xd3");

    test1(&mut sha,
        b"abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
        b"\x1a\x96\x18\x2b\x50\xfb\x8c\x7e\x74\xe0\xa7\x07\x78\x8f\x55\xe9\x82\x09\xb8\xd9\x1f\xad\xe8\xf3\x2f\x8d\xd5\xcf\xf7\xbf\x21\xf5\x4e\xe5\xf1\x95\x50\x82\x5a\x6e\x07\x00\x30\x51\x9e\x94\x42\x63\xac\x1c\x67\x65\x28\x70\x65\x62\x1f\x9f\xcb\x32\x01\x72\x3e\x32\x23\xb6\x3a\x46\xc2\x93\x8a\xa9\x53\xba\x84\x01\xd0\xea\x77\xb8\xd2\x64\x90\x77\x55\x66\x40\x7b\x95\x67\x3c\x0f\x4c\xc1\xce\x9f\xd9\x66\x14\x8d\x7e\xfd\xff\x26\xbb\xf9\xf4\x8a\x21\xc6\xda\x35\xbf\xaa");

    test1(&mut sha,
        b"\xdc\x88\x6d\xf3\xf6\x9c\x49\x51\x3d\xe3\x62\x7e\x94\x81\xdb\x58\x71\xe8\xee\x88\xeb\x9f\x99\x61\x15\x41\x93\x0a\x8b\xc8\x85\xe0",
        b"\x93\x68\xf0\x15\x10\x92\x44\xeb\x02\x47\xfa\x3a\x0e\x57\xf5\x2e\xa7\xd9\xeb\xa2\x3d\xae\x7a\x19\x7f\x0a\x29\xe9\x22\x55\x06\x05\x98\x16\xb7\x84\x48\xb6\x49\x7a\x76\xeb\x96\x2d\xb3\xf8\x4d\x37\x60\xf1\xfe\xb4\xbd\xc1\xfd\x4a\xc9\x4e\x91\x7a\xc2\xea\x5e\x4f\x38\x37\x4a\xa5\x6e\x4f\x47\x67\xb8\xd7\x83\x1b\x2d\x51\x49\x5a\xb8\xea\xb7\xc9\x82\x20\xaf\x13\x41\x5a\x59\xbb\x7c\x17\x7a\xcd\x62\x8e\xf0\xff\xe3\x6c\xeb\x18\x59\x5d\x14\x4c\xbf\x25\xef\xc0\x6c\xd9\x56\xa5\x78\x20\x6e\xa8\xf9\x14\x5e\xf9\xce\x19\x50\x6a\x9d\x04\x4e\xc7\x00\x79\x9f\xa1");

    test1(&mut sha,
        b"\x8d\x80\x01\xe2\xc0\x96\xf1\xb8\x8e\x7c\x92\x24\xa0\x86\xef\xd4\x79\x7f\xbf\x74\xa8\x03\x3a\x2d\x42\x2a\x2b\x6b\x8f\x67\x47\xe4",
        b"\xe1\x7e\xab\x0d\xa4\x04\xf9\xb6\xac\xc0\x84\x97\x2f\xc5\x79\xe8\x6d\xaa\x76\x10\xa5\xe1\x7c\x23\x2f\x79\x19\x83\x96\xfd\x01\xc2\x4c\x34\xbb\x54\xf4\xb0\x1e\xf7\x40\xb4\x25\x33\x4a\x55\xdd\x24\x81\x3d\xc8\xea\x86\xf5\x6e\xf7\x27\x67\x26\x2b\xf2\x25\x74\x8c\xcc\x3d\x9f\x48\x6f\xfb\x72\x8f\x4e\xad\x29\x60\xc9\x6c\x3e\x44\x63\x86\xea\xce\x21\x9c\x84\x28\x16\x11\x63\x58\xb0\xf4\x2d\x7d\xff\xf7\xdd\x24\x11\xfa\x2a\x56\x79\xfd\x7a\x94\x77\x45\x75\xba\xf9\xfc\xad\x68\xa1\x9e\x30\xd1\x49\xb0\x59\xb5\x9c\x44\x6c\x4e\xdc\xa5\x9b\xc5\xa4\x79\x9d\xc4\x65\xaa\x9e\x78\x2c\xed\x9f\x21\xc5\x5d\xe2\x42\xdd\x25\xd0\xd9\xde\x60\xd0\x9f\xf8\x6a\xba\xf3\xa0\x3a\x76\x71\xb3\x05\x42\xdf\xbe\x72\xfc\x56\xed\x6d\x1a\x99\x7f\x23\x7c\xd1\xa5\x50\x9e\xb0\x4d\x61\x37\xa5\xcb\x24\x71\x3b\xa3\x60\x51\x2e\x80\x83\x8b\xe0\x55\x50\xa7\x1e\xcc\x9f\xac\x41\x77\x2c\x79\x22\x30\x09\x1b\x1a\x83\x5b\x2c\x48\xdc\x09\x7d\x59\x0d\xf0\x54\x17\xfb\x5e\x38\x68\xde\xdb\xc5\x93\xab\x17\x5f\x4b\x4d\x6d\xf2\xc7\x4e\x15\x1e\x10\x76\xc4");
}

#[test]
#[cfg(shake128)]
fn test_shake128_absorb_squeeze() {
    let mut sha = SHAKE128::new().expect("Error with new()");
    fn test1(sha: &mut SHAKE128, input: &[u8], expected_squeeze_out: &[u8]) {
        sha.init().expect("Error with init()");
        sha.absorb(input).expect("Error with absorb()");
        let mut squeeze_out = vec![0u8; expected_squeeze_out.len()];
        sha.squeeze_blocks(&mut squeeze_out).expect("Error with squeeze_blocks()");
        assert_eq!(squeeze_out, *expected_squeeze_out);
    }

    test1(&mut sha, b"",
        b"\x7f\x9c\x2b\xa4\xe8\x8f\x82\x7d\x61\x60\x45\x50\x76\x05\x85\x3e\xd7\x3b\x80\x93\xf6\xef\xbc\x88\xeb\x1a\x6e\xac\xfa\x66\xef\x26\x3c\xb1\xee\xa9\x88\x00\x4b\x93\x10\x3c\xfb\x0a\xee\xfd\x2a\x68\x6e\x01\xfa\x4a\x58\xe8\xa3\x63\x9c\xa8\xa1\xe3\xf9\xae\x57\xe2\x35\xb8\xcc\x87\x3c\x23\xdc\x62\xb8\xd2\x60\x16\x9a\xfa\x2f\x75\xab\x91\x6a\x58\xd9\x74\x91\x88\x35\xd2\x5e\x6a\x43\x50\x85\xb2\xba\xdf\xd6\xdf\xaa\xc3\x59\xa5\xef\xbb\x7b\xcc\x4b\x59\xd5\x38\xdf\x9a\x04\x30\x2e\x10\xc8\xbc\x1c\xbf\x1a\x0b\x3a\x51\x20\xea\x17\xcd\xa7\xcf\xad\x76\x5f\x56\x23\x47\x4d\x36\x8c\xcc\xa8\xaf\x00\x07\xcd\x9f\x5e\x4c\x84\x9f\x16\x7a\x58\x0b\x14\xaa\xbd\xef\xae\xe7\xee\xf4\x7c\xb0\xfc\xa9");

    test1(&mut sha, b"abc",
        b"\x58\x81\x09\x2d\xd8\x18\xbf\x5c\xf8\xa3\xdd\xb7\x93\xfb\xcb\xa7\x40\x97\xd5\xc5\x26\xa6\xd3\x5f\x97\xb8\x33\x51\x94\x0f\x2c\xc8\x44\xc5\x0a\xf3\x2a\xcd\x3f\x2c\xdd\x06\x65\x68\x70\x6f\x50\x9b\xc1\xbd\xde\x58\x29\x5d\xae\x3f\x89\x1a\x9a\x0f\xca\x57\x83\x78\x9a\x41\xf8\x61\x12\x14\xce\x61\x23\x94\xdf\x28\x6a\x62\xd1\xa2\x25\x2a\xa9\x4d\xb9\xc5\x38\x95\x6c\x71\x7d\xc2\xbe\xd4\xf2\x32\xa0\x29\x4c\x85\x7c\x73\x0a\xa1\x60\x67\xac\x10\x62\xf1\x20\x1f\xb0\xd3\x77\xcf\xb9\xcd\xe4\xc6\x35\x99\xb2\x7f\x34\x62\xbb\xa4\xa0\xed\x29\x6c\x80\x1f\x9f\xf7\xf5\x73\x02\xbb\x30\x76\xee\x14\x5f\x97\xa3\x2a\xe6\x8e\x76\xab\x66\xc4\x8d\x51\x67\x5b\xd4\x9a\xcc\x29\x08\x2f\x56\x47\x58\x4e");

    test1(&mut sha,
        b"abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
        b"\x1a\x96\x18\x2b\x50\xfb\x8c\x7e\x74\xe0\xa7\x07\x78\x8f\x55\xe9\x82\x09\xb8\xd9\x1f\xad\xe8\xf3\x2f\x8d\xd5\xcf\xf7\xbf\x21\xf5\x4e\xe5\xf1\x95\x50\x82\x5a\x6e\x07\x00\x30\x51\x9e\x94\x42\x63\xac\x1c\x67\x65\x28\x70\x65\x62\x1f\x9f\xcb\x32\x01\x72\x3e\x32\x23\xb6\x3a\x46\xc2\x93\x8a\xa9\x53\xba\x84\x01\xd0\xea\x77\xb8\xd2\x64\x90\x77\x55\x66\x40\x7b\x95\x67\x3c\x0f\x4c\xc1\xce\x9f\xd9\x66\x14\x8d\x7e\xfd\xff\x26\xbb\xf9\xf4\x8a\x21\xc6\xda\x35\xbf\xaa\x54\x56\x54\xf7\x0a\xe5\x86\xff\x10\x13\x14\x20\x77\x14\x83\xec\x92\xed\xab\x40\x8c\x76\x7b\xf4\xc5\xb4\xff\xfa\xa8\x0c\x8c\xa2\x14\xd8\x4c\x4d\xc7\x00\xd0\xc5\x06\x30\xb2\xff\xc3\x79\x3e\xa4\xd8\x72\x58\xb4\xc9\x54");

    test1(&mut sha,
        b"\xdc\x88\x6d\xf3\xf6\x9c\x49\x51\x3d\xe3\x62\x7e\x94\x81\xdb\x58\x71\xe8\xee\x88\xeb\x9f\x99\x61\x15\x41\x93\x0a\x8b\xc8\x85\xe0",
        b"\x93\x68\xf0\x15\x10\x92\x44\xeb\x02\x47\xfa\x3a\x0e\x57\xf5\x2e\xa7\xd9\xeb\xa2\x3d\xae\x7a\x19\x7f\x0a\x29\xe9\x22\x55\x06\x05\x98\x16\xb7\x84\x48\xb6\x49\x7a\x76\xeb\x96\x2d\xb3\xf8\x4d\x37\x60\xf1\xfe\xb4\xbd\xc1\xfd\x4a\xc9\x4e\x91\x7a\xc2\xea\x5e\x4f\x38\x37\x4a\xa5\x6e\x4f\x47\x67\xb8\xd7\x83\x1b\x2d\x51\x49\x5a\xb8\xea\xb7\xc9\x82\x20\xaf\x13\x41\x5a\x59\xbb\x7c\x17\x7a\xcd\x62\x8e\xf0\xff\xe3\x6c\xeb\x18\x59\x5d\x14\x4c\xbf\x25\xef\xc0\x6c\xd9\x56\xa5\x78\x20\x6e\xa8\xf9\x14\x5e\xf9\xce\x19\x50\x6a\x9d\x04\x4e\xc7\x00\x79\x9f\xa1\x41\x9b\xaf\x60\x52\xc0\xc1\xb4\x45\xf8\x35\x17\x57\xb0\xd0\x22\x87\x21\x89\xe2\xc0\x27\x3f\x82\xd9\x69\x69\x66\x3e\x55\x4d\x09");

    test1(&mut sha,
        b"\x8d\x80\x01\xe2\xc0\x96\xf1\xb8\x8e\x7c\x92\x24\xa0\x86\xef\xd4\x79\x7f\xbf\x74\xa8\x03\x3a\x2d\x42\x2a\x2b\x6b\x8f\x67\x47\xe4",
        b"\xe1\x7e\xab\x0d\xa4\x04\xf9\xb6\xac\xc0\x84\x97\x2f\xc5\x79\xe8\x6d\xaa\x76\x10\xa5\xe1\x7c\x23\x2f\x79\x19\x83\x96\xfd\x01\xc2\x4c\x34\xbb\x54\xf4\xb0\x1e\xf7\x40\xb4\x25\x33\x4a\x55\xdd\x24\x81\x3d\xc8\xea\x86\xf5\x6e\xf7\x27\x67\x26\x2b\xf2\x25\x74\x8c\xcc\x3d\x9f\x48\x6f\xfb\x72\x8f\x4e\xad\x29\x60\xc9\x6c\x3e\x44\x63\x86\xea\xce\x21\x9c\x84\x28\x16\x11\x63\x58\xb0\xf4\x2d\x7d\xff\xf7\xdd\x24\x11\xfa\x2a\x56\x79\xfd\x7a\x94\x77\x45\x75\xba\xf9\xfc\xad\x68\xa1\x9e\x30\xd1\x49\xb0\x59\xb5\x9c\x44\x6c\x4e\xdc\xa5\x9b\xc5\xa4\x79\x9d\xc4\x65\xaa\x9e\x78\x2c\xed\x9f\x21\xc5\x5d\xe2\x42\xdd\x25\xd0\xd9\xde\x60\xd0\x9f\xf8\x6a\xba\xf3\xa0\x3a\x76\x71\xb3\x05\x42\xdf\xbe\x72\xfc\x56\xed\x6d\x1a\x99\x7f\x23\x7c\xd1\xa5\x50\x9e\xb0\x4d\x61\x37\xa5\xcb\x24\x71\x3b\xa3\x60\x51\x2e\x80\x83\x8b\xe0\x55\x50\xa7\x1e\xcc\x9f\xac\x41\x77\x2c\x79\x22\x30\x09\x1b\x1a\x83\x5b\x2c\x48\xdc\x09\x7d\x59\x0d\xf0\x54\x17\xfb\x5e\x38\x68\xde\xdb\xc5\x93\xab\x17\x5f\x4b\x4d\x6d\xf2\xc7\x4e\x15\x1e\x10\x76\xc4\xcb\x87\xd8\xb7\x9d\xa8\xbf\xc5\x2e\x5e\xfc\xd3\x6c\x45\xd4\x5d\x72\x0f\x66\xeb\x67\x86\xfa\x6c\xd6\x80\xa4\x23\xcb\x5d\xed\x3c\xde\xdc\x5b\x3d\xca\x95\x43\x4b\xdc\xe8\x49\xd3\xe1\x01\xd4\xf1\xe4\x47\xcf\x56\xba\x71\xb4\x69\xed\xe7\xdb\x0f\x89\xd6\xbb\xcd\x1a\xff\xb4\xbe\x72\x26\xdc\x76\x79\xb3\x1a\x4b\xe6\x8d\x9b\x8e\xd9\xe9\xe6\xf9\xff\xa5");
}

#[test]
#[cfg(shake256)]
fn test_shake256() {
    let mut sha = SHAKE256::new().expect("Error with new()");
    fn test1(sha: &mut SHAKE256, input: &[u8], expected_hash: &[u8]) {
        sha.init().expect("Error with init()");
        sha.update(input).expect("Error with update()");
        let mut hash = vec![0u8; expected_hash.len()];
        sha.finalize(&mut hash).expect("Error with finalize()");
        assert_eq!(hash, *expected_hash);
    }

    test1(&mut sha, b"",
        b"\x46\xb9\xdd\x2b\x0b\xa8\x8d\x13\x23\x3b\x3f\xeb\x74\x3e\xeb\x24\x3f\xcd\x52\xea\x62\xb8\x1b\x82\xb5\x0c\x27\x64\x6e\xd5\x76\x2f\xd7\x5d\xc4\xdd\xd8\xc0\xf2\x00\xcb\x05\x01\x9d\x67\xb5\x92\xf6\xfc\x82\x1c\x49\x47\x9a\xb4\x86\x40\x29\x2e\xac\xb3\xb7\xc4\xbe\x14\x1e\x96\x61\x6f\xb1\x39\x57\x69\x2c\xc7\xed\xd0\xb4\x5a\xe3\xdc\x07\x22\x3c\x8e\x92\x93\x7b\xef\x84\xbc\x0e\xab\x86\x28\x53\x34\x9e\xc7\x55\x46\xf5\x8f\xb7\xc2\x77\x5c\x38\x46\x2c\x50\x10\xd8\x46");

    test1(&mut sha, b"abc",
        b"\x48\x33\x66\x60\x13\x60\xa8\x77\x1c\x68\x63\x08\x0c\xc4\x11\x4d\x8d\xb4\x45\x30\xf8\xf1\xe1\xee\x4f\x94\xea\x37\xe7\x8b\x57\x39\xd5\xa1\x5b\xef\x18\x6a\x53\x86\xc7\x57\x44\xc0\x52\x7e\x1f\xaa\x9f\x87\x26\xe4\x62\xa1\x2a\x4f\xeb\x06\xbd\x88\x01\xe7\x51\xe4\x13\x85\x14\x12\x04\xf3\x29\x97\x9f\xd3\x04\x7a\x13\xc5\x65\x77\x24\xad\xa6\x4d\x24\x70\x15\x7b\x3c\xdc\x28\x86\x20\x94\x4d\x78\xdb\xcd\xdb\xd9\x12\x99\x3f\x09\x13\xf1\x64\xfb\x2c\xe9\x51\x31\xa2\xd0");

    test1(&mut sha,
        b"abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
        b"\x4d\x8c\x2d\xd2\x43\x5a\x01\x28\xee\xfb\xb8\xc3\x6f\x6f\x87\x13\x3a\x79\x11\xe1\x8d\x97\x9e\xe1\xae\x6b\xe5\xd4\xfd\x2e\x33\x29\x40\xd8\x68\x8a\x4e\x6a\x59\xaa\x80\x60\xf1\xf9\xbc\x99\x6c\x05\xac\xa3\xc6\x96\xa8\xb6\x62\x79\xdc\x67\x2c\x74\x0b\xb2\x24\xec\x37\xa9\x2b\x65\xdb\x05\x39\xc0\x20\x34\x55\xf5\x1d\x97\xcc\xe4\xcf\xc4\x91\x27\xd7\x26\x0a\xfc\x67\x3a\xf2\x08\xba\xf1\x9b\xe2\x12\x33\xf3\xde\xbe\x78\xd0\x67\x60\xcf\xa5\x51\xee\x1e\x07\x91\x41\xd4");

    test1(&mut sha,
        b"\xdc\x88\x6d\xf3\xf6\x9c\x49\x51\x3d\xe3\x62\x7e\x94\x81\xdb\x58\x71\xe8\xee\x88\xeb\x9f\x99\x61\x15\x41\x93\x0a\x8b\xc8\x85\xe0",
        b"\x00\x64\x8a\xfb\xc5\xe6\x51\x64\x9d\xb1\xfd\x82\x93\x6b\x00\xdb\xbc\x12\x2f\xb4\xc8\x77\x86\x0d\x38\x5c\x49\x50\xd5\x6d\xe7\xe0\x96\xd6\x13\xd7\xa3\xf2\x7e\xd8\xf2\x63\x34\xb0\xcc\xc1\x40\x7b\x41\xdc\xcb\x23\xdf\xaa\x52\x98\x18\xd1\x12\x5c\xd5\x34\x80\x92\x52\x43\x66\xb8\x5f\xab\xb9\x7c\x6c\xd1\xe6\x06\x6f\x45\x9b\xcc\x56\x6d\xa8\x7e\xc9\xb7\xba\x36\x79\x2d\x11\x8a\xc3\x9a\x4c\xce\xf6\x19\x2b\xbf\x3a\x54\xaf\x18\xe5\x7b\x0c\x14\x61\x01\xf6\xae\xaa\x82\x2b\xc4\xb4\xc9\x70\x8b\x09\xf0\xb3\xba\xb4\x1b\xcc\xe9\x64\xd9\x99\xd1\x10\x7b\xd7\xc2");

    test1(&mut sha,
        b"\x8d\x80\x01\xe2\xc0\x96\xf1\xb8\x8e\x7c\x92\x24\xa0\x86\xef\xd4\x79\x7f\xbf\x74\xa8\x03\x3a\x2d\x42\x2a\x2b\x6b\x8f\x67\x47\xe4",
        b"\x2e\x97\x5f\x6a\x8a\x14\xf0\x70\x4d\x51\xb1\x36\x67\xd8\x19\x5c\x21\x9f\x71\xe6\x34\x56\x96\xc4\x9f\xa4\xb9\xd0\x8e\x92\x25\xd3\xd3\x93\x93\x42\x51\x52\xc9\x7e\x71\xdd\x24\x60\x1c\x11\xab\xcf\xa0\xf1\x2f\x53\xc6\x80\xbd\x3a\xe7\x57\xb8\x13\x4a\x9c\x10\xd4\x29\x61\x58\x69\x21\x7f\xdd\x58\x85\xc4\xdb\x17\x49\x85\x70\x3a\x6d\x6d\xe9\x4a\x66\x7e\xac\x30\x23\x44\x3a\x83\x37\xae\x1b\xc6\x01\xb7\x6d\x7d\x38\xec\x3c\x34\x46\x31\x05\xf0\xd3\x94\x9d\x78\xe5\x62\xa0\x39\xe4\x46\x95\x48\xb6\x09\x39\x5d\xe5\xa4\xfd\x43\xc4\x6c\xa9\xfd\x6e\xe2\x9a\xda\x5e\xfc\x07\xd8\x4d\x55\x32\x49\x45\x0d\xab\x4a\x49\xc4\x83\xde\xd2\x50\xc9\x33\x8f\x85\xcd\x93\x7a\xe6\x6b\xb4\x36\xf3\xb4\x02\x6e\x85\x9f\xda\x1c\xa5\x71\x43\x2f\x3b\xfc\x09\xe7\xc0\x3c\xa4\xd1\x83\xb7\x41\x11\x1c\xa0\x48\x3d\x0e\xda\xbc\x03\xfe\xb2\x3b\x17\xee\x48\xe8\x44\xba\x24\x08\xd9\xdc\xfd\x01\x39\xd2\xe8\xc7\x31\x01\x25\xae\xe8\x01\xc6\x1a\xb7\x90\x0d\x1e\xfc\x47\xc0\x78\x28\x17\x66\xf3\x61\xc5\xe6\x11\x13\x46\x23\x5e\x1d\xc3\x83\x25\x66\x6c");
}

#[test]
#[cfg(shake256)]
fn test_shake256_absorb_squeeze() {
    let mut sha = SHAKE256::new().expect("Error with new()");
    fn test1(sha: &mut SHAKE256, input: &[u8], expected_squeeze_out: &[u8]) {
        sha.init().expect("Error with init()");
        sha.absorb(input).expect("Error with absorb()");
        let mut squeeze_out = vec![0u8; expected_squeeze_out.len()];
        sha.squeeze_blocks(&mut squeeze_out).expect("Error with squeeze_blocks()");
        assert_eq!(squeeze_out, *expected_squeeze_out);
    }

    test1(&mut sha, b"",
        b"\x46\xb9\xdd\x2b\x0b\xa8\x8d\x13\x23\x3b\x3f\xeb\x74\x3e\xeb\x24\x3f\xcd\x52\xea\x62\xb8\x1b\x82\xb5\x0c\x27\x64\x6e\xd5\x76\x2f\xd7\x5d\xc4\xdd\xd8\xc0\xf2\x00\xcb\x05\x01\x9d\x67\xb5\x92\xf6\xfc\x82\x1c\x49\x47\x9a\xb4\x86\x40\x29\x2e\xac\xb3\xb7\xc4\xbe\x14\x1e\x96\x61\x6f\xb1\x39\x57\x69\x2c\xc7\xed\xd0\xb4\x5a\xe3\xdc\x07\x22\x3c\x8e\x92\x93\x7b\xef\x84\xbc\x0e\xab\x86\x28\x53\x34\x9e\xc7\x55\x46\xf5\x8f\xb7\xc2\x77\x5c\x38\x46\x2c\x50\x10\xd8\x46\xc1\x85\xc1\x51\x11\xe5\x95\x52\x2a\x6b\xcd\x16\xcf\x86\xf3\xd1\x22\x10\x9e\x3b\x1f\xdd");

    test1(&mut sha, b"abc",
        b"\x48\x33\x66\x60\x13\x60\xa8\x77\x1c\x68\x63\x08\x0c\xc4\x11\x4d\x8d\xb4\x45\x30\xf8\xf1\xe1\xee\x4f\x94\xea\x37\xe7\x8b\x57\x39\xd5\xa1\x5b\xef\x18\x6a\x53\x86\xc7\x57\x44\xc0\x52\x7e\x1f\xaa\x9f\x87\x26\xe4\x62\xa1\x2a\x4f\xeb\x06\xbd\x88\x01\xe7\x51\xe4\x13\x85\x14\x12\x04\xf3\x29\x97\x9f\xd3\x04\x7a\x13\xc5\x65\x77\x24\xad\xa6\x4d\x24\x70\x15\x7b\x3c\xdc\x28\x86\x20\x94\x4d\x78\xdb\xcd\xdb\xd9\x12\x99\x3f\x09\x13\xf1\x64\xfb\x2c\xe9\x51\x31\xa2\xd0\x9a\x3e\x6d\x51\xcb\xfc\x62\x27\x20\xd7\xa7\x5c\x63\x34\xe8\xa2\xd7\xec\x71\xa7\xcc\x29");

    test1(&mut sha,
        b"abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
        b"\x4d\x8c\x2d\xd2\x43\x5a\x01\x28\xee\xfb\xb8\xc3\x6f\x6f\x87\x13\x3a\x79\x11\xe1\x8d\x97\x9e\xe1\xae\x6b\xe5\xd4\xfd\x2e\x33\x29\x40\xd8\x68\x8a\x4e\x6a\x59\xaa\x80\x60\xf1\xf9\xbc\x99\x6c\x05\xac\xa3\xc6\x96\xa8\xb6\x62\x79\xdc\x67\x2c\x74\x0b\xb2\x24\xec\x37\xa9\x2b\x65\xdb\x05\x39\xc0\x20\x34\x55\xf5\x1d\x97\xcc\xe4\xcf\xc4\x91\x27\xd7\x26\x0a\xfc\x67\x3a\xf2\x08\xba\xf1\x9b\xe2\x12\x33\xf3\xde\xbe\x78\xd0\x67\x60\xcf\xa5\x51\xee\x1e\x07\x91\x41\xd4\x9d\xd3\xef\x7e\x18\x2b\x15\x24\xdf\x82\xea\x1c\xef\xe1\xc6\xc3\x96\x61\x75\xf0\x22\x8d");

    test1(&mut sha,
        b"\xdc\x88\x6d\xf3\xf6\x9c\x49\x51\x3d\xe3\x62\x7e\x94\x81\xdb\x58\x71\xe8\xee\x88\xeb\x9f\x99\x61\x15\x41\x93\x0a\x8b\xc8\x85\xe0",
        b"\x00\x64\x8a\xfb\xc5\xe6\x51\x64\x9d\xb1\xfd\x82\x93\x6b\x00\xdb\xbc\x12\x2f\xb4\xc8\x77\x86\x0d\x38\x5c\x49\x50\xd5\x6d\xe7\xe0\x96\xd6\x13\xd7\xa3\xf2\x7e\xd8\xf2\x63\x34\xb0\xcc\xc1\x40\x7b\x41\xdc\xcb\x23\xdf\xaa\x52\x98\x18\xd1\x12\x5c\xd5\x34\x80\x92\x52\x43\x66\xb8\x5f\xab\xb9\x7c\x6c\xd1\xe6\x06\x6f\x45\x9b\xcc\x56\x6d\xa8\x7e\xc9\xb7\xba\x36\x79\x2d\x11\x8a\xc3\x9a\x4c\xce\xf6\x19\x2b\xbf\x3a\x54\xaf\x18\xe5\x7b\x0c\x14\x61\x01\xf6\xae\xaa\x82\x2b\xc4\xb4\xc9\x70\x8b\x09\xf0\xb3\xba\xb4\x1b\xcc\xe9\x64\xd9\x99\xd1\x10\x7b\xd7\xc2");

    test1(&mut sha,
        b"\x8d\x80\x01\xe2\xc0\x96\xf1\xb8\x8e\x7c\x92\x24\xa0\x86\xef\xd4\x79\x7f\xbf\x74\xa8\x03\x3a\x2d\x42\x2a\x2b\x6b\x8f\x67\x47\xe4",
        b"\x2e\x97\x5f\x6a\x8a\x14\xf0\x70\x4d\x51\xb1\x36\x67\xd8\x19\x5c\x21\x9f\x71\xe6\x34\x56\x96\xc4\x9f\xa4\xb9\xd0\x8e\x92\x25\xd3\xd3\x93\x93\x42\x51\x52\xc9\x7e\x71\xdd\x24\x60\x1c\x11\xab\xcf\xa0\xf1\x2f\x53\xc6\x80\xbd\x3a\xe7\x57\xb8\x13\x4a\x9c\x10\xd4\x29\x61\x58\x69\x21\x7f\xdd\x58\x85\xc4\xdb\x17\x49\x85\x70\x3a\x6d\x6d\xe9\x4a\x66\x7e\xac\x30\x23\x44\x3a\x83\x37\xae\x1b\xc6\x01\xb7\x6d\x7d\x38\xec\x3c\x34\x46\x31\x05\xf0\xd3\x94\x9d\x78\xe5\x62\xa0\x39\xe4\x46\x95\x48\xb6\x09\x39\x5d\xe5\xa4\xfd\x43\xc4\x6c\xa9\xfd\x6e\xe2\x9a\xda\x5e\xfc\x07\xd8\x4d\x55\x32\x49\x45\x0d\xab\x4a\x49\xc4\x83\xde\xd2\x50\xc9\x33\x8f\x85\xcd\x93\x7a\xe6\x6b\xb4\x36\xf3\xb4\x02\x6e\x85\x9f\xda\x1c\xa5\x71\x43\x2f\x3b\xfc\x09\xe7\xc0\x3c\xa4\xd1\x83\xb7\x41\x11\x1c\xa0\x48\x3d\x0e\xda\xbc\x03\xfe\xb2\x3b\x17\xee\x48\xe8\x44\xba\x24\x08\xd9\xdc\xfd\x01\x39\xd2\xe8\xc7\x31\x01\x25\xae\xe8\x01\xc6\x1a\xb7\x90\x0d\x1e\xfc\x47\xc0\x78\x28\x17\x66\xf3\x61\xc5\xe6\x11\x13\x46\x23\x5e\x1d\xc3\x83\x25\x66\x6c\x68\x1b\x30\xdd\xc4\xe6\x83\x8b\x0f\x23\x58\x7e\x06\x5f\x4a\x2b\xed\xc9\x6c\x97\x68\x44");
}
