/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.juneau.commons.utils;

import static org.apache.juneau.commons.reflect.ReflectionUtils.*;
import static org.apache.juneau.commons.utils.CollectionUtils.*;
import static org.apache.juneau.commons.utils.Utils.*;

import java.lang.annotation.*;
import java.lang.reflect.*;
import java.util.*;
import java.util.stream.*;

/**
 * Annotation utilities.
 */
public class AnnotationUtils {

	/**
	 * Checks if two annotations are equal using the criteria for equality presented in the {@link Annotation#equals(Object)} API docs.
	 *
	 * @param a1 the first Annotation to compare, {@code null} returns {@code false} unless both are {@code null}
	 * @param a2 the second Annotation to compare, {@code null} returns {@code false} unless both are {@code null}
	 * @return {@code true} if the two annotations are {@code equal} or both {@code null}
	 */
	public static boolean equals(Annotation a1, Annotation a2) {
		if (a1 == a2)
			return true;
		if (a1 == null || a2 == null)
			return false;

		var t1 = a1.annotationType();
		var t2 = a2.annotationType();

		if (! t1.equals(t2))
			return false;

		return ! getAnnotationMethods(t1).anyMatch(x -> ! memberEquals(x.getReturnType(), safeSupplier(() -> x.invoke(a1)), safeSupplier(() -> x.invoke(a2))));
	}

	/**
	 * Generate a hash code for the given annotation using the algorithm presented in the {@link Annotation#hashCode()} API docs.
	 *
	 * @param a the Annotation for a hash code calculation is desired, not {@code null}
	 * @return the calculated hash code
	 * @throws RuntimeException if an {@code Exception} is encountered during annotation member access
	 * @throws IllegalStateException if an annotation method invocation returns {@code null}
	 */
	public static int hash(Annotation a) {
		return getAnnotationMethods(a.annotationType()).mapToInt(x -> hashMember(x.getName(), safeSupplier(() -> x.invoke(a)))).sum();
	}

	/**
	 * Returns a stream of nested annotations in a repeated annotation if the specified annotation is a repeated annotation,
	 * or a singleton stream with the same annotation if not.
	 *
	 * <p>
	 * This method is a stream-based alternative to splitting repeated annotations that avoids creating intermediate arrays.
	 *
	 * <p>
	 * <b>Example:</b>
	 * <p class='bjava'>
	 * 	<jc>// Given an annotation that may be repeatable</jc>
	 * 	Annotation <jv>annotation</jv> = ...;
	 *
	 * 	<jc>// Stream individual annotations (expanded if repeatable)</jc>
	 * 	streamRepeated(<jv>annotation</jv>)
	 * 		.forEach(<jv>a</jv> -&gt; System.<jsf>out</jsf>.println(<jv>a</jv>));
	 * </p>
	 *
	 * @param a The annotation to split.
	 * @return A stream of nested annotations, or a singleton stream with the same annotation if it's not repeated.
	 * 	Never <jk>null</jk>.
	 */
	public static Stream<Annotation> streamRepeated(Annotation a) {
		var ci = info(a.annotationType());
		var mi = ci.getRepeatedAnnotationMethod();
		if (nn(mi)) {
			Annotation[] annotations = mi.invoke(a);
			return Arrays.stream(annotations);
		}
		return Stream.of(a);
	}

	private static boolean annotationArrayMemberEquals(Annotation[] a1, Annotation[] a2) {
		if (a1.length != a2.length)
			return false;
		for (var i = 0; i < a1.length; i++)
			if (neq(a1[i], a2[i]))
				return false;
		return true;
	}

	private static boolean arrayMemberEquals(Class<?> componentType, Object o1, Object o2) {
		if (componentType.isAnnotation())
			return annotationArrayMemberEquals((Annotation[])o1, (Annotation[])o2);
		if (componentType.equals(Byte.TYPE))
			return Arrays.equals((byte[])o1, (byte[])o2);
		if (componentType.equals(Short.TYPE))
			return Arrays.equals((short[])o1, (short[])o2);
		if (componentType.equals(Integer.TYPE))
			return Arrays.equals((int[])o1, (int[])o2);
		if (componentType.equals(Character.TYPE))
			return Arrays.equals((char[])o1, (char[])o2);
		if (componentType.equals(Long.TYPE))
			return Arrays.equals((long[])o1, (long[])o2);
		if (componentType.equals(Float.TYPE))
			return Arrays.equals((float[])o1, (float[])o2);
		if (componentType.equals(Double.TYPE))
			return Arrays.equals((double[])o1, (double[])o2);
		if (componentType.equals(Boolean.TYPE))
			return Arrays.equals((boolean[])o1, (boolean[])o2);
		return Arrays.equals((Object[])o1, (Object[])o2);
	}

	private static int arrayMemberHash(Class<?> componentType, Object o) {
		if (componentType.equals(Byte.TYPE))
			return Arrays.hashCode((byte[])o);
		if (componentType.equals(Short.TYPE))
			return Arrays.hashCode((short[])o);
		if (componentType.equals(Integer.TYPE))
			return Arrays.hashCode((int[])o);
		if (componentType.equals(Character.TYPE))
			return Arrays.hashCode((char[])o);
		if (componentType.equals(Long.TYPE))
			return Arrays.hashCode((long[])o);
		if (componentType.equals(Float.TYPE))
			return Arrays.hashCode((float[])o);
		if (componentType.equals(Double.TYPE))
			return Arrays.hashCode((double[])o);
		if (componentType.equals(Boolean.TYPE))
			return Arrays.hashCode((boolean[])o);
		return Arrays.hashCode((Object[])o);
	}

	private static Stream<Method> getAnnotationMethods(Class<? extends Annotation> type) {
		return l(type.getDeclaredMethods()).stream();
	}

	private static int hashMember(String name, Object value) {
		int part1 = name.hashCode() * 127;
		if (value == null)
			return part1;
		if (isArray(value))
			return part1 ^ arrayMemberHash(value.getClass().getComponentType(), value);
		if (value instanceof Annotation value2)
			return part1 ^ hash(value2);
		return part1 ^ value.hashCode();
	}

	private static boolean memberEquals(Class<?> type, Object o1, Object o2) {
		if (o1 == o2)
			return true;
		if (o1 == null || o2 == null)
			return false;
		if (type.isArray())
			return arrayMemberEquals(type.getComponentType(), o1, o2);
		if (type.isAnnotation())
			return eq((Annotation)o1, (Annotation)o2);
		return o1.equals(o2);
	}
}