/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.flink.agents.api.context;

import org.apache.flink.agents.api.Event;
import org.apache.flink.agents.api.configuration.ReadableConfiguration;
import org.apache.flink.agents.api.metrics.FlinkAgentsMetricGroup;
import org.apache.flink.agents.api.resource.Resource;
import org.apache.flink.agents.api.resource.ResourceType;

import java.util.Map;

/**
 * A context object used during action execution. It is responsible for collecting output events
 * generated by the action and providing access to state.
 */
public interface RunnerContext {
    /**
     * Sends an event.
     *
     * @param event the event to be sent
     */
    void sendEvent(Event event);

    /**
     * Gets the short-term memory.
     *
     * @return MemoryObject the root of the short-term memory
     * @throws Exception if the underlying state backend cannot be accessed
     */
    MemoryObject getShortTermMemory() throws Exception;

    /**
     * Gets the metric group for Flink Agents.
     *
     * @return the metric group shared across all actions.
     */
    FlinkAgentsMetricGroup getAgentMetricGroup();

    /**
     * Gets the individual metric group dedicated for each action.
     *
     * @return the individual metric group specific to the current action.
     */
    FlinkAgentsMetricGroup getActionMetricGroup();

    /**
     * Gets a resource by name and type.
     *
     * @param name the resource name
     * @param type the resource type
     * @return the resource instance
     * @throws Exception if the resource cannot be found or created
     */
    Resource getResource(String name, ResourceType type) throws Exception;

    /**
     * Gets the configuration for Flink Agents.
     *
     * @return the configuration for Flink Agents.
     */
    ReadableConfiguration getConfig();

    /**
     * Get the configuration of the action.
     *
     * @return the configuration of the action.
     */
    Map<String, Object> getActionConfig();

    /**
     * Get the option value of the action config.
     *
     * @return the option value of the action config.
     */
    Object getActionConfigValue(String key);
}
