/*
 * Decompiled with CFR 0.152.
 */
package org.apache.activemq.artemis.core.server.cluster.quorum;

import java.lang.invoke.MethodHandles;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import org.apache.activemq.artemis.api.core.ActiveMQException;
import org.apache.activemq.artemis.api.core.Pair;
import org.apache.activemq.artemis.api.core.SimpleString;
import org.apache.activemq.artemis.api.core.TransportConfiguration;
import org.apache.activemq.artemis.api.core.client.ClusterTopologyListener;
import org.apache.activemq.artemis.api.core.client.TopologyMember;
import org.apache.activemq.artemis.core.client.impl.ClientSessionFactoryInternal;
import org.apache.activemq.artemis.core.client.impl.TopologyMemberImpl;
import org.apache.activemq.artemis.core.protocol.core.Channel;
import org.apache.activemq.artemis.core.protocol.core.Packet;
import org.apache.activemq.artemis.core.protocol.core.impl.wireformat.QuorumVoteMessage;
import org.apache.activemq.artemis.core.protocol.core.impl.wireformat.QuorumVoteReplyMessage;
import org.apache.activemq.artemis.core.server.ActiveMQComponent;
import org.apache.activemq.artemis.core.server.ActiveMQServerLogger;
import org.apache.activemq.artemis.core.server.cluster.ClusterControl;
import org.apache.activemq.artemis.core.server.cluster.ClusterController;
import org.apache.activemq.artemis.core.server.cluster.quorum.Quorum;
import org.apache.activemq.artemis.core.server.cluster.quorum.QuorumVote;
import org.apache.activemq.artemis.core.server.cluster.quorum.QuorumVoteHandler;
import org.apache.activemq.artemis.core.server.cluster.quorum.QuorumVoteServerConnect;
import org.apache.activemq.artemis.core.server.cluster.quorum.Vote;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class QuorumManager
implements ClusterTopologyListener,
ActiveMQComponent {
    private static final Logger logger = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    private static final int VOTE_RESPONSE_TIMEOUT;
    private final ExecutorService executor;
    private final ClusterController clusterController;
    private final Map<String, Quorum> quorums = new HashMap<String, Quorum>();
    private final Map<QuorumVote, VoteRunnableHolder> voteRunnables = new HashMap<QuorumVote, VoteRunnableHolder>();
    private final Map<SimpleString, QuorumVoteHandler> handlers = new HashMap<SimpleString, QuorumVoteHandler>();
    private boolean started = false;
    private int maxClusterSize = 0;

    public QuorumManager(ExecutorService threadPool, ClusterController clusterController) {
        this.clusterController = Objects.requireNonNull(clusterController);
        this.executor = Objects.requireNonNull(threadPool);
    }

    public void start() throws Exception {
        if (this.started) {
            return;
        }
        this.started = true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void stop() throws Exception {
        if (!this.started) {
            return;
        }
        Map<QuorumVote, VoteRunnableHolder> map = this.voteRunnables;
        synchronized (map) {
            this.started = false;
            for (VoteRunnableHolder voteRunnableHolder : this.voteRunnables.values()) {
                for (VoteRunnable runnable : voteRunnableHolder.runnables) {
                    runnable.close();
                }
            }
        }
        for (Quorum quorum : this.quorums.values()) {
            quorum.close();
        }
        this.quorums.clear();
    }

    public boolean isStarted() {
        return this.started;
    }

    public void registerQuorum(Quorum quorum) {
        this.quorums.put(quorum.getName(), quorum);
        quorum.setQuorumManager(this);
    }

    public void unRegisterQuorum(Quorum quorum) {
        this.quorums.remove(quorum.getName());
    }

    public void nodeUP(TopologyMember topologyMember, boolean last) {
        int newClusterSize = this.clusterController.getDefaultClusterSize();
        this.maxClusterSize = newClusterSize > this.maxClusterSize ? newClusterSize : this.maxClusterSize;
        for (Quorum quorum : this.quorums.values()) {
            quorum.nodeUp(this.clusterController.getDefaultClusterTopology());
        }
    }

    public void nodeDown(long eventUID, String nodeID) {
        for (Quorum quorum : this.quorums.values()) {
            quorum.nodeDown(this.clusterController.getDefaultClusterTopology(), eventUID, nodeID);
        }
    }

    public boolean hasPrimary(String nodeID, int quorumSize, int voteTimeout, TimeUnit voteTimeoutUnit) {
        Objects.requireNonNull(nodeID, "nodeID");
        if (!this.started) {
            throw new IllegalStateException("QuorumManager must start first");
        }
        int size = quorumSize == -1 ? this.maxClusterSize : quorumSize;
        QuorumVoteServerConnect quorumVote = new QuorumVoteServerConnect(size, nodeID);
        boolean noPrimary = this.awaitVoteComplete(quorumVote, voteTimeout, voteTimeoutUnit);
        return !noPrimary;
    }

    public boolean isStillActive(String nodeID, TransportConfiguration connector, int quorumSize, int voteTimeout, TimeUnit voteTimeoutUnit) {
        Objects.requireNonNull(nodeID, "nodeID");
        Objects.requireNonNull(connector, "connector");
        if (!this.started) {
            throw new IllegalStateException("QuorumManager must start first");
        }
        int size = quorumSize == -1 ? this.maxClusterSize : quorumSize;
        QuorumVoteServerConnect quorumVote = new QuorumVoteServerConnect(size, nodeID, true, connector.toString());
        return this.awaitVoteComplete(quorumVote, voteTimeout, voteTimeoutUnit);
    }

    private boolean awaitVoteComplete(QuorumVoteServerConnect quorumVote, int voteTimeout, TimeUnit voteTimeoutUnit) {
        int maxClusterSize = this.maxClusterSize;
        this.vote(quorumVote);
        if (maxClusterSize > 1) {
            try {
                quorumVote.await(voteTimeout, voteTimeoutUnit);
            }
            catch (InterruptedException interruption) {
                ActiveMQServerLogger.LOGGER.quorumVoteAwaitInterrupted();
            }
        } else {
            ActiveMQServerLogger.LOGGER.ignoringQuorumVote(maxClusterSize);
        }
        this.voteComplete(quorumVote);
        return quorumVote.getDecision();
    }

    public int getMaxClusterSize() {
        return this.maxClusterSize;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void vote(QuorumVote quorumVote) {
        ArrayList<VoteRunnable> runnables = new ArrayList<VoteRunnable>();
        Map<QuorumVote, VoteRunnableHolder> map = this.voteRunnables;
        synchronized (map) {
            if (!this.started) {
                return;
            }
            ActiveMQServerLogger.LOGGER.initiatingQuorumVote(quorumVote.getName());
            for (TopologyMemberImpl tm : this.clusterController.getDefaultClusterTopology().getMembers()) {
                if (tm.getNodeId().equals(this.clusterController.getNodeID().toString())) continue;
                Pair pair = tm.getConnector();
                TransportConfiguration serverTC = (TransportConfiguration)pair.getA();
                VoteRunnable voteRunnable = new VoteRunnable(serverTC, quorumVote);
                runnables.add(voteRunnable);
            }
            int votes = runnables.size();
            ActiveMQServerLogger.LOGGER.requestedQuorumVotes(votes);
            if (votes > 0) {
                this.voteRunnables.put(quorumVote, new VoteRunnableHolder(quorumVote, runnables, votes));
                for (VoteRunnable runnable : runnables) {
                    this.executor.submit(runnable);
                }
            } else {
                quorumVote.allVotesCast(this.clusterController.getDefaultClusterTopology());
            }
        }
    }

    private Vote vote(SimpleString handler, Vote vote) {
        logger.trace("Vote for {}", (Object)handler);
        QuorumVoteHandler quorumVoteHandler = this.handlers.get(handler);
        return quorumVoteHandler.vote(vote);
    }

    private void voteComplete(QuorumVoteServerConnect quorumVote) {
        VoteRunnableHolder holder = this.voteRunnables.remove(quorumVote);
        if (holder != null) {
            for (VoteRunnable runnable : holder.runnables) {
                runnable.close();
            }
        }
    }

    public void registerQuorumHandler(QuorumVoteHandler quorumVoteHandler) {
        this.handlers.put(quorumVoteHandler.getQuorumName(), quorumVoteHandler);
    }

    public String toString() {
        return QuorumManager.class.getSimpleName() + "(server=" + this.clusterController.getIdentity() + ")";
    }

    private QuorumVoteHandler getVoteHandler(SimpleString handler) {
        if (handler.equals((Object)QuorumVoteServerConnect.OLD_PRIMARY_FAILOVER_VOTE)) {
            handler = QuorumVoteServerConnect.PRIMARY_FAILOVER_VOTE;
        }
        return this.handlers.get(handler);
    }

    public void handleQuorumVote(Channel clusterChannel, Packet packet) {
        QuorumVoteHandler voteHandler;
        QuorumVoteMessage quorumVoteMessage = (QuorumVoteMessage)packet;
        SimpleString handlerUsed = quorumVoteMessage.getHandler();
        if (quorumVoteMessage.getHandler().equals((Object)QuorumVoteServerConnect.OLD_PRIMARY_FAILOVER_VOTE)) {
            quorumVoteMessage.setHandler(QuorumVoteServerConnect.PRIMARY_FAILOVER_VOTE);
        }
        if ((voteHandler = this.getVoteHandler(quorumVoteMessage.getHandler())) == null) {
            logger.trace("Could not find {}", (Object)quorumVoteMessage.getHandler());
            ActiveMQServerLogger.LOGGER.noVoteHandlerConfigured(handlerUsed);
            return;
        }
        quorumVoteMessage.decode(voteHandler);
        ActiveMQServerLogger.LOGGER.receivedQuorumVoteRequest(quorumVoteMessage.getVote().toString());
        logger.trace("Receiving handler = {}", (Object)quorumVoteMessage.getHandler());
        Vote vote = this.vote(quorumVoteMessage.getHandler(), quorumVoteMessage.getVote());
        ActiveMQServerLogger.LOGGER.sendingQuorumVoteResponse(vote.toString());
        clusterChannel.send((Packet)new QuorumVoteReplyMessage(handlerUsed, vote));
    }

    private Vote sendQuorumVote(ClusterControl clusterControl, SimpleString handler, SimpleString oldHandlerName, Vote vote) {
        try {
            ClientSessionFactoryInternal sessionFactory = clusterControl.getSessionFactory();
            String remoteAddress = sessionFactory.getConnection().getRemoteAddress();
            ActiveMQServerLogger.LOGGER.sendingQuorumVoteRequest(remoteAddress, vote.toString());
            QuorumVoteReplyMessage replyMessage = null;
            Channel clusterChannel = clusterControl.getClusterChannel().get();
            if (clusterChannel.requireSpecialVotingHandling()) {
                logger.trace("Using special handling mechanism for Voting to cope with previous version");
                replyMessage = QuorumManager.voteFormerVersions(handler, oldHandlerName, vote, clusterChannel);
            } else {
                logger.trace("Using REGULAR voting");
            }
            if (replyMessage == null) {
                replyMessage = (QuorumVoteReplyMessage)clusterChannel.sendBlocking((Packet)new QuorumVoteMessage(handler, vote), (byte)-3);
            }
            logger.trace("Got reply message {}", (Object)replyMessage);
            QuorumVoteHandler voteHandler = this.getVoteHandler(replyMessage.getHandler());
            replyMessage.decodeRest(voteHandler);
            Vote voteResponse = replyMessage.getVote();
            ActiveMQServerLogger.LOGGER.receivedQuorumVoteResponse(remoteAddress, voteResponse.toString());
            return voteResponse;
        }
        catch (ActiveMQException e) {
            logger.debug("{}", (Object)e.getMessage(), (Object)e);
            return null;
        }
    }

    private static QuorumVoteReplyMessage voteFormerVersions(SimpleString handler, SimpleString oldHandlerName, Vote vote, Channel clusterChannel) throws ActiveMQException {
        QuorumVoteReplyMessage replyMessage = (QuorumVoteReplyMessage)clusterChannel.sendBlocking((Packet)new QuorumVoteMessage(handler, vote), -1, (byte)-3, (long)VOTE_RESPONSE_TIMEOUT, false);
        logger.trace("This is the reply message from the current version = {}", (Object)replyMessage);
        if (replyMessage == null && oldHandlerName != null) {
            replyMessage = (QuorumVoteReplyMessage)clusterChannel.sendBlocking((Packet)new QuorumVoteMessage(oldHandlerName, vote), -1, (byte)-3, (long)VOTE_RESPONSE_TIMEOUT, false);
            logger.trace("This is the reply message from the older version = {}", (Object)replyMessage);
        }
        return replyMessage;
    }

    private void votingComplete(QuorumVote quorumVote) {
        VoteRunnableHolder voteRunnableHolder = this.voteRunnables.get(quorumVote);
        if (voteRunnableHolder != null) {
            voteRunnableHolder.voteComplete();
        }
    }

    static {
        int parsedTimeout = 2000;
        try {
            parsedTimeout = Integer.parseInt(System.getProperty(Quorum.class.getName() + ".VOTE_RESPONSE_TIMEOUT", "2000"));
        }
        catch (Exception e) {
            logger.warn(e.getMessage(), (Throwable)e);
        }
        VOTE_RESPONSE_TIMEOUT = parsedTimeout;
    }

    private final class VoteRunnableHolder {
        private final QuorumVote quorumVote;
        private final List<VoteRunnable> runnables;
        private int size;

        private VoteRunnableHolder(QuorumVote quorumVote, List<VoteRunnable> runnables, int size) {
            this.quorumVote = quorumVote;
            this.runnables = runnables;
            this.size = size;
        }

        public synchronized void voteComplete() {
            --this.size;
            if (this.size <= 0) {
                this.quorumVote.allVotesCast(QuorumManager.this.clusterController.getDefaultClusterTopology());
            }
        }
    }

    private final class VoteRunnable
    implements Runnable {
        private final TransportConfiguration serverTC;
        private final QuorumVote quorumVote;
        private ClusterControl clusterControl;

        private VoteRunnable(TransportConfiguration serverTC, QuorumVote quorumVote) {
            this.serverTC = serverTC;
            this.quorumVote = quorumVote;
        }

        @Override
        public void run() {
            try {
                if (!QuorumManager.this.started) {
                    return;
                }
                this.clusterControl = QuorumManager.this.clusterController.connectToNode(this.serverTC);
                this.clusterControl.authorize();
                Vote vote = this.quorumVote.connected();
                if (vote.isRequestServerVote()) {
                    vote = QuorumManager.this.sendQuorumVote(this.clusterControl, this.quorumVote.getName(), this.quorumVote.getOldName(), vote);
                    this.quorumVote.vote(vote);
                } else {
                    this.quorumVote.vote(vote);
                }
            }
            catch (Exception e) {
                Vote vote = this.quorumVote.notConnected();
                this.quorumVote.vote(vote);
            }
            finally {
                try {
                    if (this.clusterControl != null) {
                        this.clusterControl.close();
                    }
                }
                catch (Exception exception) {}
                QuorumManager.this.votingComplete(this.quorumVote);
            }
        }

        public void close() {
            if (this.clusterControl != null) {
                this.clusterControl.close();
            }
        }
    }
}

