// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.
// See the LICENSE file in the project root for more information.

using System.Diagnostics;
using System.IO;
using System.Xml.Xsl.Runtime;

namespace System.Xml.Xsl.Qil
{
    /// <summary>
    /// Data structure for storing whitespace rules generated by xsl:strip-space and xsl:preserve-space
    /// </summary>
    internal class WhitespaceRule
    {
        private string _localName;
        private string _namespaceName;
        private bool _preserveSpace;

        /// <summary>
        /// Allow derived classes to construct empty whitespace rule.
        /// </summary>
        protected WhitespaceRule()
        {
        }

        /// <summary>
        /// Construct new whitespace rule.
        /// </summary>
        public WhitespaceRule(string localName, string namespaceName, bool preserveSpace)
        {
            Init(localName, namespaceName, preserveSpace);
        }

        /// <summary>
        /// Initialize whitespace rule after it's been constructed.
        /// </summary>
        protected void Init(string localName, string namespaceName, bool preserveSpace)
        {
            _localName = localName;
            _namespaceName = namespaceName;
            _preserveSpace = preserveSpace;
        }

        /// <summary>
        /// Local name of the element.
        /// </summary>
        public string LocalName
        {
            get { return _localName; }
            set { _localName = value; }
        }

        /// <summary>
        /// Namespace name (uri) of the element.
        /// </summary>
        public string NamespaceName
        {
            get { return _namespaceName; }
            set { _namespaceName = value; }
        }

        /// <summary>
        /// True, if this element is whitespace-preserving.
        /// False, if this element is whitespace-stripping.
        /// </summary>
        public bool PreserveSpace
        {
            get { return _preserveSpace; }
        }

        /// <summary>
        /// Serialize the object to BinaryWriter.
        /// </summary>
        public void GetObjectData(XmlQueryDataWriter writer)
        {
            Debug.Assert(this.GetType() == typeof(WhitespaceRule), "Serialization of WhitespaceRule subclasses is not implemented");
            // string localName;
            writer.WriteStringQ(_localName);
            // string namespaceName;
            writer.WriteStringQ(_namespaceName);
            // bool preserveSpace;
            writer.Write(_preserveSpace);
        }

        /// <summary>
        /// Deserialize the object from BinaryReader.
        /// </summary>
        public WhitespaceRule(XmlQueryDataReader reader)
        {
            // string localName;
            _localName = reader.ReadStringQ();
            // string namespaceName;
            _namespaceName = reader.ReadStringQ();
            // bool preserveSpace;
            _preserveSpace = reader.ReadBoolean();
        }
    }
}
