/*
  expression.hh, copyright (c) 2006 by Vincent Fourmond: 
  The main header file for SCalc.
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details (in the COPYING file).
  
*/

namespace SCalc {
  // I like Javadoc-like comments the best

  class Expression;
  class FuncDef;
  class SyntaxError;

  /** 
      \brief The result of an SCalc::Session::eval().
  
      The base class for all the things the parser can return. As the parser
      can return quite a whole bunch of different things, it is not wise
      to use SCalc::Expression as the carrier for all. Instead, here comes this
      small wrapper that enables one to check the return type of everything.
      A ParserResult can be:
      - a SCalc::Expression, when the string evaluated to a normal expression
      - a SCalc::FuncDef, when the strings evaluated to a function definition
      - a SCalc::SyntaxError, when a syntax error was detected.
  */
  class ParserResult {
    /// The base session we're talking about.
    Session * sess;
  public:
    /// Constructors/desctructors:
    ParserResult(Session * s) {sess = s;};
    virtual ~ParserResult() {;};

    /// \brief The SCalc::Session object used
    /// 
    /// Returns the SCalc::Session in which the expression
    /// was parsed.
    Session * session() { return sess;};

    // A whole set of what am I functions, all returning false 
    // (so that the children don't actually need to redefine anything.
    
    
    //@{
    /// \name Conversion functions
    ///
    /// ParserResult as such is not directly usable: you need to
    /// check its actual type and convert it to an appropriate
    /// subtype with these functions.

    /// \brief Is it a SCalc::Expression ?
    /// 
    /// Returns true if the object is a SCalc::Expression
    /// or one of its children.
    virtual int is_expression() { return 0;}

    /// \brief Converts to SCalc::Expression
    /// 
    /// Returns a SCalc::Expression object or NULL if the object isn't
    /// an expression.
    Expression * to_expression() { 
      if(is_expression())
	return (Expression *) this;
      else
	return NULL;
    };


    /// \brief Is it a SCalc::SyntaxError ?
    /// 
    /// Returns true if the object is a SCalc::SyntaxError
    /// or one of its children.
    virtual int is_syntax_error() { return 0;}

    /// \brief Converts to SCalc::SyntaxError
    /// 
    /// Returns a SCalc::SyntaxError object or NULL if the object doesn't
    /// represent a syntax error.
    SyntaxError * to_syntax_error() { 
      if(is_syntax_error())
	return (SyntaxError *) this;
      else
	return NULL;
    };

    /// \brief Is it a SCalc::FuncDef ?
    /// 
    /// Returns true if the object is a SCalc::FuncDef
    /// or one of its children.
    virtual int is_func_def() { return 0;}

    /// \brief Converts to SCalc::FuncDef
    /// 
    /// Returns a SCalc::FuncDef object or NULL if the object doesn't
    /// represent a function definition.
    FuncDef * to_func_def() { 
      if(is_func_def())
	return (FuncDef *) this;
      else
	return NULL;
    };

    //@}


    /// Pretty printing of the result ?
    virtual std::string pretty_print() = 0;

    /// Whether or not we can freely delete this result = nearly always
    /// yes, unless it is a named function
    virtual int can_delete() { return 1;};
  };


  /**
     \brief An expression !

     This class represents a valid expression. It is a pure virtual
     class, but all its descendants are internals of SCalc.
   */
  class Expression : public ParserResult{
  public:
    Expression(Session * s) : ParserResult(s) {;};
    virtual ~Expression() {;};

    /// Yes, this is an expression:
    virtual int is_expression() { return 1;};

    /** 
	\brief Evaluate the expression

	\param values the values of the indeterminates.
	\param s internal parameter.

	This function evaluates the expression using values for
	the values of the indeterminates. The second parameter is only
	used internally, you should never need it.

	values is an array of variable values. You can get their name
	with the SCalc::Session::varnames() function. This way, you can
	setup the correspondance only once for a whole vector.

	\see SCalc::Session::varnames()
    */
    virtual double evaluate(const double * values, 
			    const double * s = NULL) = 0;
    
    /// \brief Dumps the contents of the expression to a stream.
    /// 
    /// Used essentially for debugging.
    /// \deprecated Not implemented everywhere. You'll have more luck with
    /// pretty_print().
    virtual void dump(::std::ostream & stream = ::std::cerr);

    /// \brief Variable used by an expression
    /// 
    /// Returns a set containing the index of each variable
    /// used by the expression.
    /// \see SCalc::Session::varnames()
    virtual std::set<int> used_variables() { std::set<int> a; return a;};

    /// Tells if this expression can be evaluated
    int evaluable() { return session()->evaluable(this);};
    /// If it can be evaluated, returns the given value
    double evaluate();

    /// If Null, then the value is zero... Will be used for
    /// simplification, when I get to that point...
    virtual int is_null() { return 0;};

    /// If Id, then the value is always 1.
    virtual int is_id() { return 0;};

    /// If const, then the value is always 1.
    virtual int is_const() { return 0;};

    /// True if no syntax errors. When looking for the result of
    /// an eval, always look for this !
    virtual int is_valid() { return 1;};

    /**
       \brief Derive an expression with regards to one variable

       \param id the index of the variable by which we'll derive

       Returns a \b new Expression containing the derivative of
       the expression with regard to variable number id. You are
       responsible for taking care of freeing the result if
       necessary. You might want to simplify() the returned
       expression, as this is not done by default.
       
       \see simplify() SCalc::Session::varnames()
    */
    virtual Expression * derive(int id);

    /// Returns a freshly allocated copy of the expression.
    virtual Expression * copy() { return NULL;};

    /** 
	\brief Pretty prints to a string
	
	Pretty-prints the expression to a string. Does not
	attempt to simplify anything.
    */
    virtual std::string pretty_print() = 0;


    /**
       \brief Simplifies the expression
       
       Returns a copy of the expression with small simplifications
       performed. The simplification engine is very basic for now.
       It detects only some pretty obvious simplifications.

       The returned value is a fresh copy that should be
       \b deleted when not used anymore.
    */
    virtual Expression * simplify() { return copy();};

    /** 
	\brief Evaluates the Expression for different parameters
	
	\param nb the number of conditions
	\param target where the results will be stored
	\param variables array of pointers to arrays storing the conditions
	
	This function evaluates the expression nb times for the conditions
	given in variables. The results are stored in target.

	\see evaluate()
    */
    virtual double * mass_evaluate(int nb, double * target, const double **variables);

    /// \name Operations on expressions
    ///
    /// A set of function to manipulate expressions algebraically.
    /// All these functions take ownership of the arguments you give them,
    /// and those will be freed when you free the result.
    
    /*@{*/
    static Expression * add(Expression *, Expression* );
    static Expression * sub(Expression *, Expression* );
    static Expression * mul(Expression *, Expression* );
    static Expression * div(Expression *, Expression* );
    /*@}*/

  };


};

