import unittest
import warnings

from gi.repository import GtkSource

from iotas.list_formatter import check_and_extend_list, increase_indentation, decrease_indentation

warnings.filterwarnings("ignore", "version")


class Test(unittest.TestCase):

    buffer: GtkSource.Buffer

    def test_check_and_extend_list(self) -> None:
        buffer = self.__reset()
        buffer.insert(buffer.get_start_iter(), "\n")
        check_and_extend_list(buffer)
        self.assertEqual(self.__get_text(), "\n")

        for char in "-+*":
            buffer = self.__reset()
            buffer.insert(buffer.get_start_iter(), f"{char} test")
            check_and_extend_list(buffer)
            self.assertEqual(self.__get_text(), f"{char} test\n{char} ")

            buffer = self.__reset()
            buffer.insert(buffer.get_start_iter(), f"  {char} test")
            check_and_extend_list(buffer)
            self.assertEqual(self.__get_text(), f"  {char} test\n  {char} ")

        for separator in ".)":
            buffer = self.__reset()
            buffer.insert(buffer.get_start_iter(), f"1{separator} test")
            check_and_extend_list(buffer)
            self.assertEqual(self.__get_text(), f"1{separator} test\n2{separator} ")

            buffer = self.__reset()
            buffer.insert(buffer.get_start_iter(), f"  1{separator} test")
            check_and_extend_list(buffer)
            self.assertEqual(self.__get_text(), f"  1{separator} test\n  2{separator} ")

            buffer = self.__reset()
            buffer.insert(buffer.get_start_iter(), f"a{separator} test")
            check_and_extend_list(buffer)
            self.assertEqual(self.__get_text(), f"a{separator} test\nb{separator} ")

            buffer = self.__reset()
            buffer.insert(buffer.get_start_iter(), f"  a{separator} test")
            check_and_extend_list(buffer)
            self.assertEqual(self.__get_text(), f"  a{separator} test\n  b{separator} ")

    def test_decrease_indentation(self) -> None:
        buffer = self.__reset()
        decrease_indentation(buffer)
        self.assertEqual(self.__get_text(), "")

        buffer = self.__reset()
        buffer.insert(buffer.get_start_iter(), "\tplain")
        decrease_indentation(buffer)
        self.assertEqual(self.__get_text(), "plain")

        buffer = self.__reset()
        buffer.insert(buffer.get_start_iter(), "  - text")
        decrease_indentation(buffer)
        self.assertEqual(self.__get_text(), "- text")

        buffer = self.__reset()
        buffer.insert(buffer.get_start_iter(), "  - text\n  - text\n  - text")
        buffer.select_range(buffer.get_start_iter(), buffer.get_end_iter())
        decrease_indentation(buffer)
        self.assertEqual(self.__get_text(), "- text\n- text\n- text")

    def test_increase_indentation(self) -> None:
        buffer = self.__reset()
        increase_indentation(buffer)
        self.assertEqual(self.__get_text(), "\t")

        buffer = self.__reset()
        buffer.insert(buffer.get_start_iter(), "plain")
        increase_indentation(buffer)
        self.assertEqual(self.__get_text(), "plain\t")

        buffer = self.__reset()
        buffer.insert(buffer.get_start_iter(), "plain")
        buffer.place_cursor(self.buffer.get_start_iter())
        increase_indentation(buffer)
        self.assertEqual(self.__get_text(), "\tplain")

        buffer = self.__reset()
        buffer.insert(buffer.get_start_iter(), "- text")
        increase_indentation(buffer)
        self.assertEqual(self.__get_text(), "  - text")

        buffer = self.__reset()
        buffer.insert(buffer.get_start_iter(), "- text\n- text\n- text")
        buffer.select_range(buffer.get_start_iter(), buffer.get_end_iter())
        increase_indentation(buffer)
        self.assertEqual(self.__get_text(), "  - text\n  - text\n  - text")

    def __get_text(self) -> str:
        if self.buffer is None:
            return ""
        else:
            start = self.buffer.get_start_iter()
            end = self.buffer.get_end_iter()
            return self.buffer.get_text(start, end, False)

    def __reset(self) -> GtkSource.Buffer:
        self.buffer = GtkSource.Buffer.new()
        return self.buffer
